/* ---------------------------------------------------------------------------
 * Module de gestion du temps en heure, minutes, secondes. */

#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <string.h>
#include <time.h>

/* Définition d'une structure pour stocker un temps : */
typedef struct {
    int h, m, s;
} Temps;

/* Affichage d'un temps sur un fichier texte
 * Entrées : un temps t, out : fichier texte supposé ouvert en écriture
 * Sorties : écriture du temps sur out
 * Exemple d'utilisation : (affichage sur l'écran)
 *   affTemps(t, stdout); */
void affTemps(Temps t, FILE * out)
{
    fprintf(out, "%dh%02d'%02d''", t.h, t.m, t.s);
}

/* Addition des deux temps t1 et t2 */
Temps addTemps(Temps t1, Temps t2)
{
    Temps res = { 0, 0, 0 };

    res.s = t1.s + t2.s;
    if (res.s >= 60) {
        res.s = res.s - 60;
        res.m = 1;
    }
    res.m = res.m + t1.m + t2.m;
    if (res.m >= 60) {
        res.m = res.m - 60;
        res.h = 1;
    }
    res.h = res.h + t1.h + t2.h;

    return res;
}

/* Différence gd - pt (on suppose gd >= pt) */
Temps diffTemps(Temps gd, Temps pt)
{
    Temps res = { 0, 0, 0 };

    res.s = gd.s - pt.s;
    if (res.s < 0) {
        res.s = res.s + 60;
        res.m = -1;
    }
    res.m = res.m + gd.m - pt.m;
    if (res.m < 0) {
        res.m = res.m + 60;
        res.h = -1;
    }
    res.h = res.h + gd.h - pt.h;

    return res;
}

/* Comparaison de deux temps, retourne -1 si t1 < t2, 0 si t1 = t2 et 
 * 1 si t1 > t2 */
int compareTemps(Temps t1, Temps t2)
{
    if (t1.h < t2.h)
        return -1;

    if (t1.h > t2.h)
        return 1;

    if (t1.m < t2.m)
        return -1

    if (t1.m > t2.m)
        return 1

    if (t1.s < t2.s)
        return -1

    if (t1.s > t2.s)
        return 1

    return 0
}

/* Retourne l'heure système */
Temps Heure(void)
{
    struct tm *temps;
    time_t timer;
    Temps res;

    timer = time(NULL);
    temps = localtime(&timer);
    res.h = temps->tm_hour;
    res.m = temps->tm_min;
    res.s = temps->tm_sec;

    return res
}

/* ---------------------------------------------------------------------------
 * Module de gestion des coureurs.  */

/* Nom par défaut du fichier de coureurs */
#define FICHIER_COUREURS "coureurs.dat"

/* Définition d'une structure pour stocker un coureur :
 *   = une ligne du fichier texte
 *   = un enregistrement du fichier binaire */
typedef struct {
    int dossard;
    char nom[31];               /* du coureur */
    char equipe[4];
    char pays[4];
    Temps temps;                /* au classement général */
} Coureur;

/* Compte le nombre de caractères UTF-8 dans la chaîne, pour corriger l'affichage */
int countUTF8chars(char *c)
{
    int n = 0;

    while (*c != 0)
        if (((unsigned char) *c) > 127) {
            n++;
            c++;
            c++;
        } else
            c++;
    return n
}

/* Affichage d'un coureur sur un fichier texte
 * Entrées : un coureur, out : fichier texte supposé ouvert en écriture
 * Sorties : écriture du coureur sur out
 * Exemple d'utilisation : (affichage sur l'écran)
 *   affCoureur(c, stdout); */
void affCoureur(Coureur c, FILE * out)
{
    int largeurNom = 40 + countUTF8chars(c.nom);

    fprintf(out, "(%03d) %-*s (%3s, %3s) : ", c.dossard, largeurNom, c.nom, c.equipe, c.pays);
    affTemps(c.temps, out);
    putchar('\n');
}

/* Affichage d'un fichier de coureurs sur un fichier texte
 * Entrées : nomCoureurs, fichier binaire de coureurs ;
 *           out : fichier texte supposé ouvert en écriture
 * Sorties : écriture du contenu de nomCoureurs sur out
 * Exemple d'utilisation : (affichage sur l'écran)
 *   afficheCoureurs("coureurs.dat", stdout); */
void afficheCoureurs(char nomCoureurs[], FILE * out)
{
    Coureur c;

    FILE *fCoureurs;

    fCoureurs = ffopen("rb", nomCoureurs);
    while (fread(&c, sizeof(Coureur), 1, fCoureurs) != 0)
        affCoureur(c, out);
    fclose(fCoureurs);
}

/* Recherche d'un coureur dans un fichier binaire de coureurs
 * Entrées : le numéro de dossard du coureur ;
 *           coureurs : fichier binaire supposé ouvert en lecture ;
 * Sorties : rang du coureur dans le fichier ou -1 s'il n'existe pas */
int rangCoureur(int dossard, FILE * coureurs)
{
    int rang;
    bool trouve;
    Coureur c;

    fseek(coureurs, 0, SEEK_SET);
    trouve = false;
    rang = 0;
    while ((fread(&c, sizeof(Coureur), 1, coureurs) != 0) && (!trouve)) {
        if (c.dossard == dossard)
            trouve = true;
        else
            rang++;
    }
    if (trouve)
        return rang
    else
        return -1
}

/* ---------------------------------------------------------------------------
 * Programme de conversion d'un fichier texte d'équipes en fichier binaire
 * de coureurs
 * Auteur(s)  : 
 * Groupe TP  :
 * Création   : 
 * Entrées    : un fichier texte donné sur la ligne de commande, equipes.txt 
 *              par défaut.
 *     Syntaxe du fichier :
 *  <ÉQUIPES> ::= <ÉQUIPE> +
 *  <ÉQUIPE> ::= <ABR> <NOM COMPLET DE L'ÉQUIPE>'\n'<COUREUR> +
 *  <COUREUR> ::= '\t'<DOSSARD>' '<NOM>' ''('<NATIONALITÉ>')''\n'
 *  <ABR> ::= <chaîne de 3 caractères>
 *  <NOM COMPLET DE L'ÉQUIPE> ::= <chaîne de caractères>
 *  <DOSSARD> ::= <entier>
 *  <NOM> ::= <chaîne de caractères>
 *  <NATIONALITÉ> ::= <chaîne de 3 caractères>
 * Sorties    : un fichier binaire (coureurs.dat)
 *     Le fichier binaire contient un élément par coureur sous la forme 
 *     d'un enregistrement (de taille fixe, voir la structure Coureur).
 * Avancement : 
 */

/* Nom par défaut du fichier des équipes */
#define FICHIER_EQUIPES "equipes.txt"

/* Taille maximum d'une ligne */
#define MAXLIGNE 128

/* Le caractère nul, plus facile à taper comme ça : */
#define NUL '\0'

/*
 * Affichage du mode d'emploi et sortie du programme
 * Sorties : affichage sur la sortie erreur standard */
void Usage(void)
{
    fprintf(stderr, "Usage : \n");
    fprintf(stderr, "\ttxt2dat [[EQUIPES=equipes.txt [COUREURS=coureurs.dat]]\n");
    exit(EXIT_FAILURE);
}
/* ---------------------------------------------------------------------
 * Programme principal :
 *   . Pour chaque équipe du fichier d'entrée :
 *     .. récupération de l'abréviation du nom
 *     .. pour chaque coureur de l'équipe :
 *       ... construction de la structure Coureur et enregistrement 
 *           dans le fichier de sortie (et affichage).
 */
int main(int nbArgs, char *arg[])
{
    char *nomEquipes, *nomCoureurs;
    char ligne[MAXLIGNE];       /* pour lecture du fichier des équipes */
    char *lCour;
    FILE *fEquipes;
    FILE *fCoureurs;

    switch (nbArgs) {
    case 1:                    /* pas d'arguments */
        nomEquipes = FICHIER_EQUIPES;
        nomCoureurs = FICHIER_COUREURS;
        break;
    case 2:                    /* fichier des équipes fourni en argument */
        nomEquipes = arg[1];
        nomCoureurs = FICHIER_COUREURS;
        break;
    case 3:                    /* fichier des équipes et fichier des coureurs fournis */
        nomEquipes = arg[1];
        nomCoureurs = arg[2];
        break;
    default:
        Usage();
    }

   /*** à compléter ***/
    printf("Création du fichier %s à partir du fichier %s\n", nomCoureurs, nomEquipes);

    fEquipes = fopen(nomEquipes, "r");
    fCoureurs = fopen(nomCoureurs, "wb");
    /* Pour chaque équipe : */
    lCour = fgets(ligne, MAXLIGNE, fEquipes);
    while (lCour != NULL) {
        Coureur c;

        /* récupérer le nom de l'équipe en abrégé */
        ligne[3] = NUL;
        strcpy(c.equipe, ligne);

        /* Pour chaque coureur de l'équipe : */
        lCour = fgets(ligne, MAXLIGNE, fEquipes);
        while ((lCour != NULL) && (ligne[0] == '\t')) {
            char *posNom, *posNat;
            int lgNom;

            /* Extraction du dossard */
            sscanf(ligne, "%d", &c.dossard);
            /* Extraction du nom */
            posNom = strchr(ligne, ' ') + 1;
            posNat = strchr(ligne, '(') + 1;
            lgNom = posNat - posNom - 2;
            posNom[lgNom] = NUL;
            strcpy(c.nom, posNom);

            /* Extraction de la nationalité */
            posNat[3] = NUL;
            strcpy(c.pays, posNat);

            /* Extraction du temps */
            sscanf(posNat + 4, "%dh%d'%d''", &c.temps.h, &c.temps.m, &c.temps.s);

            /* Enregistrement */
            fwrite(&c, sizeof(Coureur), 1, fCoureurs);
            affCoureur(c, stdout);

            lCour = fgets(ligne, MAXLIGNE, fEquipes);
        }
    }

    fclose(fEquipes);
    fclose(fCoureurs);

    afficheCoureurs(nomCoureurs, stdout);

    return EXIT_SUCCESS;
}
